/*
 *  globalfields.js
 *
 *  Copyright: (c) 2018 FileMaker, Inc. All rights reserved.
 *
 *  FILEMAKER CONFIDENTIAL
 *  This file includes unpublished proprietary source code of FileMaker,
 *  Inc. The copyright notice above does not evidence any actual or
 *  intended publication of such source code.
 */

var errorHandler = require('../routes/error');
var util = require('../routes/util');


/**
 * @api {PUT} /:solution/solutions/globals  setGlobalFields
 * @apiDescription set the value for Global Fields in a layout.
 * @apiName setGlobalFields
 * @apiGroup GlobalFields
 *
 * @apiUse AccessTokenHeader
 *
 * @apiParam {json} globalFields	A JSON object contains field-value pairs for global fields to set in a target layout.
 *
 * @apiParamExample {json} Request-Body-Example:
 *    { "globalFields": {
 *    	"globalFieldName1":"globalFieldValue1",
 *			"globalFieldName2":"globalFieldValue2"
 *			}
 *    }
 *
 *
 * @apiSuccess {String} errorCode "0"
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "errorCode": "0"
 *     }
 *
 * @apiUse BadRequestError
 * @apiUse UnauthorizedError
 * @apiUse FMServiceError
 *
 *
 * @apiSampleRequest /global/:solution/:layout/
 */
module.exports.setGlobalFields = function(req, res, next){
	if(!req.body['globalFields']){
		return errorHandler.handleError('BadRequest', req, res, "globalFields is missing", "10");
	}
	var data = (typeof req.body['globalFields'] === "object") ? JSON.stringify(req.body['globalFields']) : req.body['globalFields'].toString();
	if(data.trim().length===0){
		return errorHandler.handleError("BadRequest", req, res, "Data is empty.","10");
	}
	delete req.body['globalFields'];

	var remainingProps = Object.keys(req.body);
	if (remainingProps.length > 0) {
		return errorHandler.handleError("BadRequest", req, res, "Unknown parameter(s): "+remainingProps.join(','),"960");		
	}
	var ipStr = util.parseClientIp(req);
	var params = {
		'solution': req.swagger.params.database.value,
		'globalFields': data,
		'requestMethod': req.method,
		'requestPath': req.originalUrl,
		'requestIp': ipStr,
		'inputBandwidth': (req.body) ? JSON.stringify(req.body).length.toString() : "0"
	};

	var result = util.setAuthentication(req, res, params);	
	if(result){
		try{
			util.thrift_client.setGlobalFields(params,
			function(thrifError, thrifResult){
				return util.handleThrifReturn(thrifError, thrifResult, req, res);
			});
		} catch (err){
			util.thriftExceptionHandler(err, req, res);
		}
	}
};
